#include "WaterShader.h"
#include <algorithm>


WaterShader::WaterShader(ID3D11Device* device, HWND hwnd) : BowerShader(device, hwnd)
{
	initShader(L"water_vs.cso", L"water_ps.cso");
	_MoveFactor = 0.0f;
	_WaveSpeed = 0.03f;
}

WaterShader::~WaterShader()
{
	//Release the buffers
	if (matrixBuffer)
	{
		matrixBuffer->Release();
		matrixBuffer = 0;
	}

	if (waterPixelBuffer)
	{
		waterPixelBuffer->Release();
		waterPixelBuffer = 0;
	}

	if (waterVertexBuffer)
	{
		waterVertexBuffer->Release();
		waterVertexBuffer = 0;
	}

	//Release the sampler
	if (sampleState)
	{
		sampleState->Release();
		sampleState = 0;
	}

	if (layout)
	{
		layout->Release();
		layout = 0;
	}

	BaseShader::~BaseShader();
}

void WaterShader::initShader(const wchar_t* vsFilename, const wchar_t* psFilename)
{
	loadTextureVertexShader(vsFilename);
	loadPixelShader(psFilename);

	//Create the matrix buffer
	D3D11_BUFFER_DESC matrixBufferDesc;
	matrixBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	matrixBufferDesc.ByteWidth = sizeof(MatrixBufferType);
	matrixBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	matrixBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	matrixBufferDesc.MiscFlags = 0;
	matrixBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&matrixBufferDesc, NULL, &matrixBuffer);

	//Create the water pixel buffer
	D3D11_BUFFER_DESC waterPixelBufferDesc;
	waterPixelBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	waterPixelBufferDesc.ByteWidth = sizeof(WaterPixelBufferType);
	waterPixelBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	waterPixelBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	waterPixelBufferDesc.MiscFlags = 0;
	waterPixelBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&waterPixelBufferDesc, NULL, &waterPixelBuffer);

	//Create the water vertex buffer
	D3D11_BUFFER_DESC waterVertexBufferDesc;
	waterVertexBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	waterVertexBufferDesc.ByteWidth = sizeof(WaterVertexBufferType);
	waterVertexBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	waterVertexBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	waterVertexBufferDesc.MiscFlags = 0;
	waterVertexBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&waterVertexBufferDesc, NULL, &waterVertexBuffer);

	//Create the sampler
	D3D11_SAMPLER_DESC samplerDesc;
	samplerDesc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
	samplerDesc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.BorderColor[0] = 1.0f;
	samplerDesc.BorderColor[1] = 0.0f;
	samplerDesc.BorderColor[2] = 0.0f;
	samplerDesc.BorderColor[3] = 1.0f;
	samplerDesc.MipLODBias = 0.0f;
	samplerDesc.MaxAnisotropy = 1;
	samplerDesc.ComparisonFunc = D3D11_COMPARISON_ALWAYS;
	samplerDesc.MinLOD = 0;
	samplerDesc.MaxLOD = D3D11_FLOAT32_MAX;
	renderer->CreateSamplerState(&samplerDesc, &sampleState);
}

void WaterShader::setShaderParameters(ID3D11DeviceContext* deviceContext, const XMMATRIX &worldMatrix, const XMMATRIX &viewMatrix, const XMMATRIX &projectionMatrix, const XMMATRIX &reflectionMatrix, ID3D11ShaderResourceView* reflectionTexture, ID3D11ShaderResourceView* refractionTexture, ID3D11ShaderResourceView* dudvTexture, ID3D11ShaderResourceView* normalTexture, XMFLOAT3 cameraPosition, float deltaTime)
{
	_MoveFactor += _WaveSpeed * deltaTime;	//Increment the move factor to move the water waves

	D3D11_MAPPED_SUBRESOURCE mappedResource;
	XMMATRIX tworld, tview, tproj, treflect;

	tworld = XMMatrixTranspose(worldMatrix);
	tview = XMMatrixTranspose(viewMatrix);
	tproj = XMMatrixTranspose(projectionMatrix);
	treflect = XMMatrixTranspose(reflectionMatrix);

	//Map the matrix buffer
	deviceContext->Map(matrixBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	MatrixBufferType* matrixPtr = (MatrixBufferType*)mappedResource.pData;
	matrixPtr->world = tworld;
	matrixPtr->view = tview;
	matrixPtr->projection = tproj;
	deviceContext->Unmap(matrixBuffer, 0);
	deviceContext->VSSetConstantBuffers(0, 1, &matrixBuffer);

	//Map the water vertex buffer
	deviceContext->Map(waterVertexBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	WaterVertexBufferType* waterVertexPtr = (WaterVertexBufferType*)mappedResource.pData;
	waterVertexPtr->cameraPosition = XMFLOAT4(cameraPosition.x, cameraPosition.y, cameraPosition.z, 1.0f);
	waterVertexPtr->reflectionMatrix = treflect;
	waterVertexPtr->sunPosition = _SunLight->getPosition();
	deviceContext->Unmap(waterVertexBuffer, 0);
	deviceContext->VSSetConstantBuffers(1, 1, &waterVertexBuffer);

	//Map the water pixel buffer
	deviceContext->Map(waterPixelBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	WaterPixelBufferType* waterPixelPtr = (WaterPixelBufferType*)mappedResource.pData;
	waterPixelPtr->moveFactor = _MoveFactor;
	waterPixelPtr->sunColour = XMFLOAT3(_SunLight->getAmbientColour().x, _SunLight->getAmbientColour().y, _SunLight->getAmbientColour().z);
	deviceContext->Unmap(waterPixelBuffer, 0);
	deviceContext->PSSetConstantBuffers(0, 1, &waterPixelBuffer);

	//Set the pixel shader textures and samplers
	deviceContext->PSSetShaderResources(0, 1, &reflectionTexture);
	deviceContext->PSSetShaderResources(1, 1, &refractionTexture);
	deviceContext->PSSetShaderResources(2, 1, &dudvTexture);
	deviceContext->PSSetShaderResources(3, 1, &normalTexture);
	deviceContext->PSSetSamplers(0, 1, &sampleState);
}